/*
	Rutines for calculating the statistics H, I and J for test of
	serial independence. For details about these statistics see
	the document "HIJ - A set of computer routines for test of 
	serial independence" by Hans Julius Skaug.

        Written by Hans Julius Skaug, October 15. 1994

        Please direct any questions or comments to:

        email: skaug@imr.no
*/
	
#include <stdio.h>
#include <math.h>
#include "hij.h"

extern long rand();

double  f_est_1(int t,  double X[], int n, double h)
/* 	
	Returns the kernel estimate (leave-one-out and Gaussian kernel)
	of the marginal density f of the observations X[0],...,X[n-1].
	Parameters:
			t: f(X[t]) is returned
			X: the vector of observations
			n: the dimension of X
			h: the bandwidth parameter
*/
{
	int s;
	double sum = -1.0, y;

	/* 	Checking for illegal parameter values.	*/
        if( !( n>=2 && 0<=t && t<=(n-1) && h>eps ) )
	{
                fprintf(stderr,"######## f_est_1: n=%d t=%d h=%f",n,t,h);
		return;
	}
		
        for ( s=0; s <= n-1; s++ )
	{
		y = (X[s]-X[t])/h;
		sum += exp( -0.5*y*y);
	}

	return(sum*k_N01/((n-1)*h));
}

double  f_est_2(int t,  double X[], int n, double h, int m)
/*
	Returns the kernel estimate (leave-one-out and Gaussian kernel)
	of the lag-m density f of the observations X[0],...,X[n-1].
	Parameters:
			t: f(X[t],X[t-m]) is returned
			X: the vector of observations
			n: the dimension of X
			h: the bandwidth parameter
			m: the lag parameter 
*/
{
	int s;
	double  sum= -1.0, y[2];


        if( !( n>=2 && 1<=m && m<=(n-2) && m<=t && t<=(n-1) && h>eps ) )
	{
                fprintf(stderr,"##### f_est_2: n = %d t = %d m = %d h=%f",n,t,m,h);
		return;
	}

        for ( s=m; s <= n-1; s++ )
	{
		y[0] = (X[s]-X[t])/h;
		y[1] = (X[s-m]-X[t-m])/h;
		sum += exp( -0.5*( y[0]*y[0] + y[1]*y[1]));
	}
		
	return(sum/(n-m-1)*k_N01*k_N01/h/h);
}



void    hij(
		double 	X[],
		int 	n,
		double 	h,
		double  H[],
		double 	I[],
		double 	J[],
		int 	q,
		double 	tmp[],
		double 	a,
		double 	b
				)
/*
	Returns the H[m], I[m] and J[m] statistics for  m=1,...,q-1. 
	By convension H[0]=I[0]=J[0]=0.0.
        Parameters:
                        X: the vector of observations
                        n: the dimension of X
                        h: the bandwidth parameter
			H,I,J: vectors for the returned statistics
                        q: maximum lag for the statistics to be calculated
			tmp: vector of the same dimension as X (working space)
			a,b: S=[a,b]x[a,b] (see documentation)
*/	
{
        int m, t, W;
	double f_12; 


        for ( m=0; m <= q-1; m++ ) /* initialization */
	{
                H[m] = 0.0;
                I[m] = 0.0;
		J[m] = 0.0;
	}

	tmp[0] = f_est_1(0,X,n,h); 

        for ( t=1; t <= n-1; t++ )
	{
		tmp[t] = f_est_1(t,X,n,h);
	        for ( m=1; m <= min(t,q-1); m++ )
		{
			W = (a<=X[t])&&(b>=X[t])&&(a<=X[t-m])&&(b>=X[t-m]);
			f_12 = f_est_2(t,X,n,h,m);

 			J[m] += (f_12 - tmp[t]*tmp[t-m])*W;
			if( f_12 > eps && tmp[t] > eps && tmp[t-m] > eps) 
			{
			   H[m] += 2*(1.0-sqrt(tmp[t]*tmp[t-m]/f_12))*W;
			   I[m] += log( f_12/(tmp[t]*tmp[t-m]) )*W;
			}
		}
	}

        for ( m=1; m <= q-1; m++ )
	{
		H[m] /= (n-m);
                I[m] /= (n-m);
                J[m] /= (n-m);
	}
}

void    cumulative( double T[], int q )
/*
	For generation of cumulative test statistics.
*/
{
	int t;
 
	for ( t=2; t <= q-1; t++ )
		T[t] += T[t-1];         
}


void	rand_perm(int r[], int n)
/*
	Permutes the elements in r randomly.
*/
{
	int i, j, tmp;

	for ( i=1; i<=n; i++ )
	{
		j = rand()%(n-i+1);
		tmp = r[n-i];
		r[n-i] = r[j];
		r[j] = tmp;
	}
}

double sum(double M[][N], int n, int j)
/*
	The routine calculates the mean along the j'te row in 
	the matrix M.
*/
{
	int i;
	double tmp=0.0;

        if( !( 0<=j && j<=(n-1) ) )
        {
                fprintf(stderr,"######## sum: n=%d j=%d ",n,j);
                return;
        }
 
        for ( i=0; i<=n-1; i++ )
		tmp += M[j][i];

	return( tmp/(n-1) );
}

double ff_m(double M[][N], int r[], int n, int j, int m)
/*
	Calculates the sum C(j,i)*C(j-m,i-m) over i. Here
	C(i,j) = M(r[i],r[j]), where r is a permutation
	of 0,1,...,n-1.
*/
{
	int i;
	double tmp=0.0;

        if( !( (1<=m) && (m<=j) && (j<=(n-1)) ) )
        {
                fprintf(stderr,"######## ff_m: n=%d j=%d m=%d",n,j,m);
                return;
        }
 
        for ( i=m; i<=n-1; i++ )
		tmp += M[r[j]][r[i]]*M[r[j-m]][r[i-m]];

	return( tmp/(n-m-1) );
}

void calc_M(double M[][N], double X[], int n, double h)
/*	
	Assigns values to the elements in M after the rule:
	
	M(i,j) = k_h(X[i]-X[j]),	0 <= i,j <= n-1.
*/
{
	int i, j;

        for ( i=0; i<=n-1; i++ )
		M[i][i] = 0.0;		/* leave-one-out */

        for ( i=1; i<=n-1; i++ )
          for ( j=0; j<=i-1; j++ )
	  {
		M[i][j] = k_h(X[i]-X[j]);	
		M[j][i] = M[i][j];	/* symmetry */
	  }
}

void    hij_perm(
		double 	X[],
		double 	M[][N],
		double 	f[],
		int 	r[],
		int 	n,
		double  H[],
		double  I[],
		double  J[],
		int 	q,
                double  a,
                double  b
				)
/*
        Returns the H[m], I[m] and J[m] statistics for m=0,1,...,q-1. 
	By convension H[0]=I[0]=J[0]=0.0.
*/
{
        int m, t, W;
	double ff;


        for ( m=0; m <= q-1; m++ )
	{
                H[m] = 0.0;
                I[m] = 0.0;
		J[m] = 0.0;
	}

        for ( t=1; t <= n-1; t++ )
	{
	        for ( m=1; m <= min(t,q-1); m++ )
		{
                        W = (a<=X[r[t]])&&(b>=X[r[t]])&&
				(a<=X[r[t-m]])&&(b>=X[r[t-m]]);
                        if( W ) 
                                ff = ff_m(M,r,n,t,m);

 			J[m] += (ff - f[r[t]]*f[r[t-m]])*W;

			if( ff > eps && f[r[t]] > eps && f[r[t-m]] > eps) 
			{
			   	H[m] += 2*(1.0-sqrt(f[r[t]]*f[r[t-m]]/ff))*W;
			   	I[m] += log( ff/(f[r[t]]*f[r[t-m]]) )*W;
			}
		}
	}

        for ( m=1; m <= q-1; m++ )
	{
                H[m] /= (n-m);
                I[m] /= (n-m);
                J[m] /= (n-m);
	}
}

void 	perm_test(
                double  X[],
                int     n,
                double  h,
                double  *P_H,
                double  *P_I,
                double  *P_J,
                int     q,
                double  tmp[],
                double  a,
                double  b
				)
/* 	
	Returns the approximate (conditional on the order statistic) 
	p-values for the statistics H, I and J on lag 1,...,q. 
	The p-values are found by simulation, B simulations being performed. 
	By convension P_H[0]=P_I[0]=P_J[0]=0.

        Parameters:
                        X: the vector of observations
                        n: the dimension of X
                        h: the bandwidth parameter
                        P_H,P_I,P_J: pointers to the returned p-values
			q: the lag on which the test statistics are
			   evaluated. 
			tmp: vector of dimension (q+1)*6. Used by the 
			     routine as working space
                        a,b: S=[a,b]x[a,b] 
*/

{
	int i, r[N];
	double M[N][N], f[N], *H, *I, *J, *H_p, *I_p, *J_p;


        if( n != N )
	{
                fprintf(stderr,"######## perm_test: n=%d N=%d\n",n,N);
                fprintf(stderr," See the maunual for explanation\n");
		return;
	}

	H=tmp; 
	I=tmp+(q+1); 
	J=tmp+2*(q+1); 
	H_p=tmp+3*(q+1); 
	I_p=tmp+4*(q+1); 
	J_p=tmp+5*(q+1);

	*P_H = 0.0;
	*P_I = 0.0;
	*P_J = 0.0;

	calc_M(M,X,n,h);

        for ( i=0; i <= n-1; i++ ) 	
	{
		f[i] = sum(M,n,i);
		r[i] = i;
	}

	hij_perm(X,M,f,r,n,H,I,J,q+1,a,b);
	cumulative(H,q+1);
	cumulative(I,q+1);
	cumulative(J,q+1);
	
	/* Simulate the permutation p-values */
	for ( i=1; i <= B; i++ )
	{
		rand_perm(r,n);	
		hij_perm(X,M,f,r,n,H_p,I_p,J_p,q+1,a,b);
		cumulative(H_p,q+1);
		cumulative(I_p,q+1);
		cumulative(J_p,q+1);

		*P_H += H_p[q] >= H[q];	
		*P_I += I_p[q] >= I[q];	
		*P_J += J_p[q] >= J[q];	
	}

	*P_H /= B;	
	*P_I /= B;	
	*P_J /= B;	
}
